cfxReconMode = {}
cfxReconMode.version = "1.0.0"

--[[--
VERSION HISTORY
 1.0.0 - initial version 
 
 cfxReconMode is a script that allows units to perform reconnaissance
 missions and, after detecting units on, marks them on the map with 
 markers 
 
--]]--
cfxReconMode.ups = 1 -- updates per second
cfxReconMode.scouts = {} -- units that are performing scouting. 
cfxReconMode.detectedGroups = {} -- so we know which have been detected
cfxReconMode.marksFadeAfter = 600 -- after detection, marks disappear after
                     -- this amount of seconds. -1 means no fade
                     -- 600 is ten minutes
cfxReconMode.prioList = {} -- group names that are high prio
cfxReconMode.blackList = {} -- group names athat are never detected
cfxReconMode.detectionMinRange = 6000 -- meters at ground level
cfxReconMode.detectionMaxRange = 8000 -- meters at max alt (10'000m)
cfxReconMode.maxAlt = 10000 -- alt for maxrange

cfxReconMode.callbacks = {} -- sig: cb(reason, side, scout, group)
cfxReconMode.uuidCount = 0 -- for unique marks 

function cfxReconMode.uuid()
    cfxReconMode.uuidCount = cfxReconMode.uuidCount + 1
    return cfxReconMode.uuidCount
end

function cfxReconMode.addCallback(theCB)
    table.insert(cfxReconMode.callbacks, theCB)
end

function cfxReconMode.invokeCallbacks(reason, theSide, theSout, theGroup)
    for idx, theCB in pairs(cfxReconMode.callbacks) do 
        theCB(reason, theSide, theScout, theGroup)
    end
end

function cfxReconMode.addScout(theUnit)
    if not theUnit then 
        trigger.action.outText("+++cfxRecon: nil Unit on add", 30)
        return
    end
    if type(theUnit) == "string" then 
        trigger.action.outText("+++cfxRecon: will access vby name: " .. theUnit, 30)
        local u = Unit.getByName(theUnit) 
        theUnit = u
    end 
    
    if not theUnit then 
        trigger.action.outText("+++cfxRecon: did not find unit on add", 30)
        return 
    end    
    cfxReconMode.scouts[theUnit:getName()] = theUnit
end

function cfxReconMode.removeScout(theUnit)
    if type(theUnit) == "string" then 
        theUnit = Unit:getByName(theUnit) 
    end 
    if not theUnit then return end    
    cfxReconMode.scouts[theUnit:getName()] = nil
end

function cfxReconMode.canDetect(scoutPos, theGroup, visRange)
    -- determine if a member of theGroup can be seen from 
    -- scoutPos at visRange 
    -- returns true and pos when detected
    local allUnits = theGroup:getUnits()
    for idx, aUnit in pairs(allUnits) do
        if aUnit:isExist() and aUnit:getLife() >= 1 then 
            local uPos = aUnit:getPoint()
            uPos.y = uPos.y + 3 -- raise my 3 meters
            local d = dcsCommon.distFlat(scoutPos, uPos) 
            if d < visRange then 
                -- is in visual range. do we have LOS?
                if land.isVisible(scoutPos, uPos) then 
                    -- group is visible, stop here, return true
                    return true, uPos
                end
            end
        end        
    end
    return false, nil -- nothing visible
end

function cfxReconMode.placeMarkForUnit(location, theSide, theGroup) 
    local theID = cfxReconMode.uuid()
    trigger.action.markToCoalition(
                    theID, 
                    "Contact: "..theGroup:getName(), 
                    location, 
                    theSide, 
                    false, 
                    nil)
    return theID
end

function cfxReconMode.removeMarkForArgs(args)
    local theSide = args[1]
    local theScout = args[2]
    local theGroup = args[3]
    local theID = args[4]
    
    trigger.action.removeMark(theID)
    cfxReconMode.detectedGroups[theGroup:getName()] = nil 
    
    -- invoke callbacks
    cfxReconMode.invokeCallbacks("removed", theSide, theScout, theGroup)
end 


function cfxReconMode.detectedGroup(mySide, theScout, theGroup, theLoc)
    -- put a mark on the map 
    local theID = cfxReconMode.placeMarkForUnit(theLoc, mySide, theGroup)
    
    -- schedule removal if desired 
    if cfxReconMode.marksFadeAfter > 0 then 
        args = {mySide, theScout, theGroup, theID}
        timer.scheduleFunction(cfxReconMode.removeMarkForArgs, args, timer.getTime() + cfxReconMode.marksFadeAfter)
    end
    
    -- say something
    trigger.action.outTextForCoalition(
            mySide,
            theScout:getName() .. " reports new ground contact " .. theGroup:getName(),
            30
        )
    -- play a sound 
    trigger.action.outSoundForCoalition(mySide, "UI_SCI-FI_Tone_Bright_Dry_20_stereo.wav")
    
    -- invoke callbacks
    cfxReconMode.invokeCallbacks("detected", mySide, theSout, theGroup)
end

function cfxReconMode.performReconForUnit(theScout)
    if not theScout then return end 
    if not theScout:isExist() then return end 
    -- get altitude above ground to calculate visual range 
    local alt = dcsCommon.getUnitAGL(theScout)
    local visRange = dcsCommon.lerp(cfxReconMode.detectionMinRange, cfxReconMode.detectionMaxRange, alt/cfxReconMode.maxAlt)
    local scoutPos = theScout:getPoint()
    -- figure out which groups we are looking for
    local myCoal = theScout:getCoalition()
    local enemyCoal = 1 
    if myCoal == 1 then enemyCoal = 2 end 
    
    -- iterate all enemy units until we find one 
    -- and then stop this iteration (can only detect one 
    -- group per pass)
    local enemyGroups = coalition.getGroups(enemyCoal)
    for idx, theGroup in pairs (enemyGroups) do 
        -- make sure it's a ground unit 
        local isGround = theGroup:getCategory() == 2
        if theGroup:isExist() and isGround then 
            local visible, location = cfxReconMode.canDetect(scoutPos, theGroup, visRange)
            if visible then 
                -- see if we already detected this one 
                
                if cfxReconMode.detectedGroups[theGroup:getName()] == nil then 
                    -- visible and not yet seen 
                    -- perhaps add some percent chance now 
                    -- remember that we know this group 
                    cfxReconMode.detectedGroups[theGroup:getName()] = theGroup
                    cfxReconMode.detectedGroup(myCoal, theScout, theGroup, location)
                    return -- stop, as we only detect one group per pass
                end
            end
        end
    end
end

function cfxReconMode.update()
    -- schedule next call 
    timer.scheduleFunction(cfxReconMode.update, {}, timer.getTime() + 1/cfxReconMode.ups)
    
    -- now process all scouts
    for idx, scout in pairs(cfxReconMode.scouts) do 
        cfxReconMode.performReconForUnit(scout)
    end
end

function cfxReconMode.start()
    -- start update cycle
    cfxReconMode.update()
    
    trigger.action.outText("cfx Recon version " .. cfxReconMode.version .. " started.", 30)
    return true
end

if not cfxReconMode.start() then 
    cfxReconMode = nil
end

--[[--
recon for (multiple) units that detect ground vehicles when in range and LOS, and places markers on the F10 map. Optional time-out after n. Make recon range dependent on height, add a percentage each cycle. check LOS. when flying low, higher percentage, when flying high, greater range
planes start and stop recon by adding/removing them from reconPool. each plane in the pool has their own detection pool 
make sure not to assume that the unit doing scouting is a plane, but vehicles already do recon on theior own, so...
 
Make players be able to turn on / off by using comms
 
unitDetected callback
 
whitelist/blacklist: units on blacklist will never be detected, units on whitelist (prio list) will generate special marks and messges. smokelist: smoke on detected unit immediately, but only one per group?

separate playerGUI so players can start/stop recon 
--]]--


 
 